param(
  [Parameter(Mandatory=$true)] [string]$Zip,
  [Parameter(Mandatory=$true)] [string]$Src,
  [Parameter(Mandatory=$true)] [string]$Entry
)

# Kontroly vstupu
if (!(Test-Path -LiteralPath $Zip)) { Write-Host "  [SKIP] Nenalezen archiv: $Zip"; exit 2 }
if (!(Test-Path -LiteralPath $Src)) { Write-Host "  [SKIP] Chybi zdroj: $Src"; exit 3 }

# Načti .NET knihovny (pro jistotu obě)
try { Add-Type -AssemblyName System.IO.Compression     | Out-Null } catch { [void][Reflection.Assembly]::LoadWithPartialName("System.IO.Compression") }
try { Add-Type -AssemblyName System.IO.Compression.FileSystem | Out-Null } catch { [void][Reflection.Assembly]::LoadWithPartialName("System.IO.Compression.FileSystem") }

# Normalizace entry cesty na ZIP formát (pouze '/')
$Target = ($Entry -replace '\\','/').TrimStart('/')

if (-not ([type]"System.IO.Compression.ZipArchiveMode")) {
  Write-Host "  [FAIL] Chybi .NET typy pro zip (System.IO.Compression)."
  exit 1
}

try {
  # Otevři ZIP v režimu Update
  $fs = [System.IO.File]::Open($Zip, [System.IO.FileMode]::Open, [System.IO.FileAccess]::ReadWrite, [System.IO.FileShare]::None)
  try {
    $zipObj = New-Object System.IO.Compression.ZipArchive($fs, [System.IO.Compression.ZipArchiveMode]::Update, $true)

    # Smazat všechny existující položky, které se shodují po normalizaci
    $toDelete = @()
    foreach ($e in $zipObj.Entries) {
      $norm = ($e.FullName -replace '\\','/').TrimStart('/')
      if ($norm -ieq $Target) { $toDelete += $e }
    }
    foreach ($e in $toDelete) { $e.Delete() }

    # Vytvořit novou položku s optimální kompresí
    $newEntry = $zipObj.CreateEntry($Target, [System.IO.Compression.CompressionLevel]::Optimal)

    # Zapsat obsah
    $inStream  = [System.IO.File]::OpenRead($Src)
    $outStream = $newEntry.Open()
    try {
      $inStream.CopyTo($outStream)
    } finally {
      $outStream.Dispose()
      $inStream.Dispose()
    }

    $zipObj.Dispose()
  }
  finally {
    $fs.Dispose()
  }

  exit 0
}
catch {
  Write-Host "  [FAIL] PowerShell chyba: $($_.Exception.Message)"
  exit 1
}
